<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Post;
use Auth;
use DB;
use Toastr;
use Session;
use App\CategoriePost;
use App\Tag;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class PostController extends Controller
{


    public function index()
    {
        $posts = Post::latest()->get(); 

        return view('admin.blog.posts.index', compact('posts'));
    }

    public function pending()
    {
        $posts = Post::where('is_approved', false)->get();
        return view('admin.blog.posts.pending', compact('posts'));
    }




    public function create()
    {
        $categories = CategoriePost::all();
        $tags = Tag::all();
        return view('admin.blog.posts.create', compact('categories','tags'));
    }




    public function store(Request $request)
    {
        $this->validate($request,[
            'title'       => 'required|unique:posts',
            'image'       => 'required',
            'categories'  => 'required',
            'tags'        => 'required',
        ]);

        $image = $request->file('image');
        $slug  = Str::slug($request->title);
        
        if(isset($image)){
            $currentDate = Carbon::now()->toDateString();
            $imageName= $slug.'-'.$currentDate.'-'.uniqid().'.'.$image->getClientOriginalExtension();
           
            if(!Storage::disk('public')->exists('posts')){
                Storage::disk('public')->makeDirectory('posts');}

            $img = Image::make($image->getRealPath());
            $img->stream(); // <-- Key point
            Storage::disk('local')->put('public/posts/'.$imageName, $img);   

        } else {
            $imageName = "default.png";
        }

        $post= new Post();
        $post->admin_id     = Auth::id();
        $post->title        = $request->title;
        $post->slug         = $slug;
        $post->image        = $imageName;
        $post->description  = $request->description;
        $post->status       = $request->has('status')?1:0;
        $post->is_approved  = Auth::user()->is_admin ? true : false;
        $post->save();

        $post->categories()->attach($request->categories);
        $post->tags()->attach($request->tags);

        
        Toastr::success('message', 'Post created successfully.'); 
        return redirect('/admin/post');


    }


    public function enable($id) {
        $post = Post::findOrFail($id);
            if ($post->status == 1)          
            { Post::where('id', $id)->update(['status' => 0]);
            Toastr::success('message', 'élement invisible.'); }
            else
            { Post::where('id', $id)->update(['status' => 1]);
            Toastr::success('message', 'élement visible.'); }
          
          return redirect('/admin/post');
    }





    public function approval($id)
    {
        $post = Post::findOrFail($id);

        if($post->is_approved == 1)
            { Post::where('id', $id)->update(['is_approved' => 0]);
            Toastr::success('message', 'élement inapproved.'); }
            else
            { Post::where('id', $id)->update(['is_approved' => 1]);
            Toastr::success('message', 'élement approved.'); }
        return redirect()->back();

    }



    public function edit(Post $post)
    {
        $categories = CategoriePost::all();
        $tags = Tag::all();
        return view('admin.blog.posts.edit', compact('post','categories','tags'));
}





    public function update(Request $request,$id)
    { 
          
        $this->validate($request,[
            'title'       => ['required', 'max:191', Rule::unique("posts")->ignore($id)],
            'image'       => 'image',
            'categories'  => 'required',
            'tags'        => 'required',
        ]);

        $post = Post::find($id);

        $image = $request->file('image');
        $slug  = Str::slug($request->title);
        
        if(isset($image)){
            $currentDate = Carbon::now()->toDateString();
            $imageName= $slug.'-'.$currentDate.'-'.uniqid().'.'.$image->getClientOriginalExtension();


            if(!Storage::disk('public')->exists('posts')){

                Storage::disk('public')->makeDirectory('posts');
            }

            // Delete old image
            if(Storage::disk('public')->exists('posts/'.$post->image)){
                Storage::disk('public')->delete('posts/'.$post->image);
            }

            // resizeimage
            $postImage = Image::make($image)->resize(1600,1066)->save($imageName, 90);
            Storage::disk('public')->put('posts/'.$imageName, $postImage);

        } else {
            $imageName = $post->image;
        }

        $post->admin_id     = Auth::id();
        $post->title        = $request->title;
        $post->slug         = $slug;
        $post->image        = $imageName;
        $post->description  = $request->description;
        $post->status       = $request->has('status')?1:0;
        $post->save();
        
        $post->categories()->sync($request->categories);
        $post->tags()->sync($request->tags);

        Toastr::success('message', 'update Post successfully.'); 
        return redirect('/admin/post');
    }





    public function destroy($id)
    {   $post = Post::find($id);
        if(Storage::disk('public')->exists('posts/'.$post->image)){
            Storage::disk('public')->delete('posts/'.$post->image);
        }

        $post->categories()->detach();
        $post->tags()->detach();

        $post->delete();

        Toastr::success('message', 'Deleted Post successfully.');
        return redirect('/admin/post');
    }




}
